-- **************************************************
-- Provide Moho with the name of this script object
-- **************************************************

ScriptName = "LK_LayerOpacity"

-- **************************************************
-- General information about this script
-- **************************************************

LK_LayerOpacity = {}

function LK_LayerOpacity:Name()
    return 'LK_LayerOpacity'
end

function LK_LayerOpacity:Version()
    return '0.0'
end

function LK_LayerOpacity:UILabel()
    return 'LK_LayerOpacity'
end

function LK_LayerOpacity:Creator()
    return 'Lukas Krepel'
end

function LK_LayerOpacity:Description()
    return 'Change opacity for selected layers. 100% keys are green, 0% keys are red, keys matching the preset are purple, percentage values in between are orange.'
end

function LK_LayerOpacity:ColorizeIcon()
    return true
end

-- **************************************************
-- Is Relevant / Is Enabled
-- **************************************************

function LK_LayerOpacity:IsRelevant(moho)
	if MohoMode ~= nil then
		if not MohoMode.animation then
			return false
		end
	end
    return true
end

function LK_LayerOpacity:IsEnabled(moho)
    return true
end

-- **************************************************
-- Tool options - create and respond to tool's UI
-- **************************************************
-- * Slider:
LK_LayerOpacity.DRAG_SLIDER				= MOHO.MSG_BASE
LK_LayerOpacity.DRAG_SLIDER_TEXT_INPUT	= MOHO.MSG_BASE + 1
LK_LayerOpacity.DRAG_PRESET_TEXT_INPUT	= MOHO.MSG_BASE + 2
LK_LayerOpacity.SET_ZERO				= MOHO.MSG_BASE + 3
LK_LayerOpacity.SET_HUNDRED				= MOHO.MSG_BASE + 4
LK_LayerOpacity.SET_PRESET				= MOHO.MSG_BASE + 5
LK_LayerOpacity.PREVIOUS_KEY 			= MOHO.MSG_BASE + 6
LK_LayerOpacity.NEXT_KEY 				= MOHO.MSG_BASE + 7

LK_LayerOpacity.presetValue = 65

function LK_LayerOpacity:LoadPrefs(prefs)
	LK_LayerOpacity.presetValue = prefs:GetInt("LK_LayerOpacity.presetValue", 65)
end

function LK_LayerOpacity:SavePrefs(prefs)
	prefs:SetInt("LK_LayerOpacity.presetValue", self.presetValue)
end

function LK_LayerOpacity:ResetPrefs(prefs)
	self.presetValue = 65
end


-- **************************************************
-- Layout
-- **************************************************
function LK_LayerOpacity:DoLayout(moho, layout)
	-- **********************
	-- *** Layer Opacity: ***
	-- **********************
	FO_Utilities:Divider(layout, "Layer Opacity", true)
	layout:PushH(LM.GUI.ALIGN_CENTER, 0)
	-- * 0 button:
	self.zeroButton = LM.GUI.ImageButton("ScriptResources/FO_icons/percentage_0", "0%", false, self.SET_ZERO, true)
	layout:AddChild(self.zeroButton)
	-- * Slider:
	local sliderSize = 500
	self.slider = LM.GUI.Slider(sliderSize, false, true, self.DRAG_SLIDER)
	layout:AddChild(self.slider)
	self.slider:SetRange(0, 100)
	self.slider:SetNumTicks(4)
	self.slider:SetSnapToTicks(false)
	-- * 100 button:
	self.hundredButton = LM.GUI.ImageButton("ScriptResources/FO_icons/percentage_100", "100%", false, self.SET_HUNDRED, true)
	layout:AddChild(self.hundredButton)
	layout:Pop()
	-- ***
	layout:PushH(LM.GUI.ALIGN_CENTER, 0)
	-- * Slider text input:
	self.slidertext = LM.GUI.TextControl(30, "0", self.DRAG_SLIDER_TEXT_INPUT, LM.GUI.FIELD_INT, MOHO.Localize(""))
	self.slidertext:SetWheelInc(1)
	layout:AddChild(self.slidertext)
	self.perc = LM.GUI.StaticText("%")
	layout:AddChild(self.perc)
	layout:Pop()
	-- ***************
	-- *** Preset: ***
	-- ***************
	FO_Utilities:Divider(layout, "Preset")
	layout:PushH(LM.GUI.ALIGN_CENTER, 0)
	self.favButton = LM.GUI.ImageButton("ScriptResources/FO_icons/arrow_right", "Set layer opacity to preset value", false, self.SET_PRESET, true)
	layout:AddChild(self.favButton)
	-- * Slider text input:
	self.presetText = LM.GUI.TextControl(30, "0", self.DRAG_PRESET_TEXT_INPUT, LM.GUI.FIELD_INT, MOHO.Localize(""))
	self.presetText:SetWheelInc(1)
	layout:AddChild(self.presetText)
	self.presetPerc = LM.GUI.StaticText("%")
	layout:AddChild(self.presetPerc)
	layout:Pop()
	-- *
	-- ************************
	-- *** Go to prev/next: ***
	-- ************************
	FO_Utilities:Divider(layout, "Go to previous/next layer opacity key")
	layout:PushH(LM.GUI.ALIGN_CENTER, 0)
	self.prevButton = LM.GUI.ImageButton("ScriptResources/FO_icons/arrow_left", "Go to previous layer opacity key", false, self.PREVIOUS_KEY, true)
	layout:AddChild(self.prevButton)
	self.nextButton = LM.GUI.ImageButton("ScriptResources/FO_icons/arrow_right", "Go to next layer opacity key", false, self.NEXT_KEY, true)
	layout:AddChild(self.nextButton)
	layout:Pop()
end

function LK_LayerOpacity:UpdateWidgets(moho)
	local alpha = moho.layer.fAlpha:GetValue(moho.frame)*100
	self.slider:SetValue(alpha)
	self.slidertext:SetValue(alpha)
	self.presetText:SetValue(self.presetValue)
end

function LK_LayerOpacity:SetOpacityForSelectedLayers(moho, percentage)
	for i = 0, moho.document:CountSelectedLayers()-1 do
		local layer = moho.document:GetSelectedLayer(i)
		layer.fAlpha:SetValue(moho.frame, percentage/100)
	end	
end

-- **************************************************
-- Keyboard/Mouse Control
-- **************************************************

function LK_LayerOpacity:OnMouseDown(moho, mouseEvent)
    
end

function LK_LayerOpacity:HandleMessage(moho, view, msg)
	if msg == self.DRAG_SLIDER then
		LK_LayerOpacity:SetOpacityForSelectedLayers(moho, self.slider:Value())
	elseif msg == self.DRAG_SLIDER_TEXT_INPUT then
		LK_LayerOpacity:SetOpacityForSelectedLayers(moho, self.slidertext:Value())
	elseif msg == self.SET_ZERO then
		LK_LayerOpacity:SetOpacityForSelectedLayers(moho, 0)
	elseif msg == self.SET_HUNDRED then
		LK_LayerOpacity:SetOpacityForSelectedLayers(moho, 100)
	elseif msg == self.SET_PRESET then
		LK_LayerOpacity:SetOpacityForSelectedLayers(moho, self.presetValue)
	elseif msg == self.DRAG_PRESET_TEXT_INPUT then
		LK_LayerOpacity.presetValue = self.presetText:Value()
		self:UpdateWidgets(moho)
	elseif msg == self.PREVIOUS_KEY then
		self:GoToNextKey(moho, true)
	elseif msg == self.NEXT_KEY then
		self:GoToNextKey(moho, false)
	end
	FO_Utilities:PercentageKeyColors(moho)
	moho:UpdateUI()
	MOHO.Redraw()
end

function LK_LayerOpacity:GoToNextKey(moho, backwards)
	backwards = backwards or false
	local keyedFrames = {}
	local channel = moho.layer.fAlpha
	table.insert(keyedFrames, 0)
	for markerID = 1, channel:CountKeys()-1 do
		local keyedFrame = channel:GetKeyWhen(markerID)
		table.insert(keyedFrames, keyedFrame)
		local interp = MOHO.InterpSetting:new_local()
		channel:GetKeyInterp(keyedFrame, interp)
		if interp.hold > 0 then
			local holdEndFrame = keyedFrame + interp.hold
			table.insert(keyedFrames, holdEndFrame)
		end
	end
	local gotoFrame
	if backwards then
		gotoFrame = -1000000000
	else
		gotoFrame = 1000000000
	end
	for i = 1, #keyedFrames do
		local keyedFrame = keyedFrames[i]
		if backwards then
			if keyedFrame < moho.frame and gotoFrame < keyedFrame then
				gotoFrame = keyedFrame
			end
		else
			if keyedFrame > moho.frame and gotoFrame > keyedFrame then
				gotoFrame = keyedFrame
			end
		end
	end
	if gotoFrame == -1000000000 or gotoFrame == 1000000000 then
		return
	end
	moho:SetCurFrame(gotoFrame)
end