-- **************************************************
-- Provide Moho with the name of this script object
-- **************************************************

ScriptName = "MR_Name"

-- **************************************************
-- General information about this script
-- **************************************************

MR_Name = {}

function MR_Name:Name()
	return self:Localize('UILabel')
end

function MR_Name:Version()
	return '1.8'
end

function MR_Name:UILabel()
	return self:Localize('UILabel')
end

function MR_Name:Creator()
	return 'Eugene Babich'
end

function MR_Name:Description()
	return self:Localize('Description')
end

-- **************************************************
-- Is Relevant / Is Enabled
-- **************************************************

function MR_Name:IsRelevant(moho)
	return true
end

function MR_Name:IsEnabled(moho)
	return true
end

-- **************************************************
-- Recurring Values
-- **************************************************

MR_Name.bone = true
MR_Name.layer = false
MR_Name.slots = 24
MR_Name.maxSlots = 200
MR_Name.row = 6
MR_Name.rowMin = 2
MR_Name.rowMax = 20
MR_Name.slotSize = 24
MR_Name.slotSizeMin = 8
MR_Name.slotSizeMax = 100
MR_Name.insertAtStart = false
MR_Name.needReopen = false
MR_Name.maxRows = 35
MR_Name.findText = ''
MR_Name.replaceText = ''
MR_Name.namingMode = false
MR_Name.extraMenu = false
MR_Name.swapMode = false
MR_Name.lastSlotsList = {}
MR_Name.lastSlots = 0
MR_Name.showSelectedBones = false

MR_Name.swapSlotId = 0

for i=1, MR_Name.maxSlots do
	MR_Name['name'..i] = ''
end

-- **************************************************
-- Prefs
-- **************************************************

function MR_Name:LoadPrefs(prefs)
	self.bone = prefs:GetBool("MR_Name.bone", true)
	self.layer = prefs:GetBool("MR_Name.layer", false)
	self.slots = prefs:GetInt("MR_Name.slots", 24)
	self.row = prefs:GetInt("MR_Name.row", 6)
	self.extraMenu = prefs:GetBool("MR_Name.extraMenu", false)
	self.swapMode = prefs:GetBool("MR_Name.swapMode", false)
	self.insertAtStart = prefs:GetBool("MR_Name.insertAtStart", false)
	self.slotSize = prefs:GetInt("MR_Name.slotSize", 24)
	self.findText = prefs:GetString("MR_Name.findText", '')
	self.replaceText = prefs:GetString("MR_Name.replaceText", '')
	self.showSelectedBones = prefs:GetBool("MR_Name.showSelectedBones", false)
	for i=1, MR_Name.slots do
		MR_Name['name'..i] = prefs:GetString("MR_Name.name"..i, '')
	end
end

function MR_Name:SavePrefs(prefs)
	prefs:SetBool("MR_Name.bone", self.bone)
	prefs:SetBool("MR_Name.layer", self.layer)
	prefs:SetInt("MR_Name.slots", self.slots)
	prefs:SetInt("MR_Name.row", self.row)
	prefs:SetInt("MR_Name.slotSize", self.slotSize)
	prefs:SetBool("MR_Name.extraMenu", self.extraMenu)
	prefs:SetBool("MR_Name.swapMode", self.swapMode)
	prefs:SetBool("MR_Name.insertAtStart", self.insertAtStart)
	prefs:SetString("MR_Name.findText", self.findText)
	prefs:SetString("MR_Name.replaceText", self.replaceText)
	prefs:SetBool("MR_Name.showSelectedBones", self.showSelectedBones)
	for i=1, MR_Name.slots do
		prefs:SetString("MR_Name.name"..i, MR_Name['name'..i])
	end
end

function MR_Name:ResetPrefs()
	self.bone = true
	self.layer = false
	self.slots = 24
	self.row = 6
	self.slotSize = 24
	self.extraMenu = false
	self.swapMode = false
	self.insertAtStart = false
	self.findText = ''
	self.replaceText = ''
	self.showSelectedBones = false
	for i=1, MR_Name.slots do
		MR_Name['name'..i] = ''
	end
end

-- **************************************************
-- MR_NameSetTextDialog
-- **************************************************

local MR_NameSetTextDialog = {}

function MR_NameSetTextDialog:new()
    local d = LM.GUI.SimpleDialog(MR_Name:Localize('Enter text'), MR_NameSetTextDialog)
    local l = d:GetLayout()

    l:PushV()
        d.Input = LM.GUI.TextControl(200, '', 0, LM.GUI.FIELD_TEXT, '')
        l:AddChild(d.Input, LM.GUI.ALIGN_LEFT, 0)
    l:Pop()
    return d
end

function MR_NameSetTextDialog:UpdateWidgets(moho)
	self.Input:SetValue(MR_Name['name'..MR_Name.index])
end

function MR_NameSetTextDialog:OnOK(moho)
	MR_Name['name'..MR_Name.index] = self.Input:Value()
end

-- **************************************************
-- MR_NameDialog
-- **************************************************

local MR_NameDialog = {}

MR_NameDialog.BONE = MOHO.MSG_BASE
MR_NameDialog.LAYER = MOHO.MSG_BASE + 1
MR_NameDialog.LOAD = MOHO.MSG_BASE + 2
MR_NameDialog.SAVE = MOHO.MSG_BASE + 3
MR_NameDialog.NAMING_MODE = MOHO.MSG_BASE + 4
MR_NameDialog.CLEAR_SLOTS = MOHO.MSG_BASE + 5
MR_NameDialog.SORT = MOHO.MSG_BASE + 6
MR_NameDialog.SORT_ALT = MOHO.MSG_BASE + 7
MR_NameDialog.UNDO = MOHO.MSG_BASE + 8
MR_NameDialog.SWAP_MODE = MOHO.MSG_BASE + 9
MR_NameDialog.ROW = MOHO.MSG_BASE + 10
MR_NameDialog.SLOT_SIZE = MOHO.MSG_BASE + 11
MR_NameDialog.SLOTS = MOHO.MSG_BASE + 12
MR_NameDialog.MINUS_SLOT = MOHO.MSG_BASE + 13
MR_NameDialog.PLUS_SLOT = MOHO.MSG_BASE + 14
MR_NameDialog.EXPAND = MOHO.MSG_BASE + 15
MR_NameDialog.FIND = MOHO.MSG_BASE + 16
MR_NameDialog.REPLACE = MOHO.MSG_BASE + 17
MR_NameDialog.REPLACE_BUTTON = MOHO.MSG_BASE + 18
MR_NameDialog.REPLACE_BUTTON_ALT = MOHO.MSG_BASE + 19
MR_NameDialog.SELECT_BUTTON = MOHO.MSG_BASE + 20
MR_NameDialog.SELECT_BUTTON_ALT = MOHO.MSG_BASE + 21
MR_NameDialog.SHOW_SELECTED_BONES = MOHO.MSG_BASE + 22
MR_NameDialog.CLEAR_PREFIX = MOHO.MSG_BASE + 23
MR_NameDialog.CLEAR_PREFIX_ALT = MOHO.MSG_BASE + 24
MR_NameDialog.CLEAR_SUFFIX = MOHO.MSG_BASE + 25
MR_NameDialog.CLEAR_SUFFIX_ALT = MOHO.MSG_BASE + 26
MR_NameDialog.INSERT_AT_START = MOHO.MSG_BASE + 27
MR_NameDialog.NAME_BUTTON = MOHO.MSG_BASE + 28
MR_NameDialog.NAME_BUTTON_ALT = MOHO.MSG_BASE + 29 + MR_Name.maxSlots

MR_NameDialog.nameButton = {}

function MR_NameDialog:new()
	local d = LM.GUI.SimpleDialog(MR_Name:Localize('UILabel'), MR_NameDialog)
	local l = d:GetLayout()
	local padding = -11
	
	l:PushH()
		d.boneButton = LM.GUI.ImageButton('ScriptResources/mr_name/mr_bone', MR_Name:Localize('Bone'), true, d.BONE, false)
		l:AddChild(d.boneButton, LM.GUI.ALIGN_LEFT, 0)
		
		l:AddPadding(padding)
		
		d.layerButton = LM.GUI.ImageButton('ScriptResources/mr_name/mr_layer', MR_Name:Localize('Layer'), true, d.LAYER, false)
		l:AddChild(d.layerButton, LM.GUI.ALIGN_LEFT, 0)
		
		l:AddPadding(padding)
		
		l:AddChild(LM.GUI.Divider(true), LM.GUI.ALIGN_FILL)
		
		l:AddPadding(padding)
		
		d.loadButton = LM.GUI.ImageButton('ScriptResources/mr_name/mr_load', MR_Name:Localize('Load'), false, d.LOAD, false)
		l:AddChild(d.loadButton, LM.GUI.ALIGN_LEFT, 0)
		
		l:AddPadding(padding)
		
		d.saveButton = LM.GUI.ImageButton('ScriptResources/mr_name/mr_save', MR_Name:Localize('Save'), false, d.SAVE, false)
		l:AddChild(d.saveButton, LM.GUI.ALIGN_LEFT, 0)
		
		l:AddPadding(padding)
		
		l:AddChild(LM.GUI.Divider(true), LM.GUI.ALIGN_FILL)
		
		l:AddPadding(padding)
		
		d.insertAtStartButton = LM.GUI.ImageButton('ScriptResources/mr_name/mr_insert_at_start', MR_Name:Localize('Insert text at start'), true, d.INSERT_AT_START, false)
		l:AddChild(d.insertAtStartButton, LM.GUI.ALIGN_LEFT, 0)

		l:AddPadding(padding)

		l:AddChild(LM.GUI.Divider(true), LM.GUI.ALIGN_FILL)
		
		l:AddPadding(padding)
		
		d.namingModeButton = LM.GUI.ImageButton('ScriptResources/mr_name/mr_naming_mode', MR_Name:Localize('Naming mode'), true, d.NAMING_MODE, false)
		l:AddChild(d.namingModeButton, LM.GUI.ALIGN_LEFT, 0)
		
		l:AddPadding(padding)
		
		d.swapModeButton = LM.GUI.ImageButton('ScriptResources/mr_name/mr_swap', MR_Name:Localize('Swap mode'), true, d.SWAP_MODE, false)
		l:AddChild(d.swapModeButton, LM.GUI.ALIGN_LEFT, 0)
		
		l:AddPadding(padding)
		
		d.clearSlotsButton = LM.GUI.ImageButton('ScriptResources/mr_name/mr_clear_slots', MR_Name:Localize('Clear slots'), false, d.CLEAR_SLOTS, false)
		l:AddChild(d.clearSlotsButton, LM.GUI.ALIGN_LEFT, 0)
		
		l:AddPadding(padding)
		
		d.sortButton = LM.GUI.ImageButton('ScriptResources/mr_name/mr_sort', MR_Name:Localize('Sort alphabetically'), false, d.SORT, false)
		d.sortButton:SetAlternateMessage(d.SORT_ALT)
		l:AddChild(d.sortButton, LM.GUI.ALIGN_LEFT, 0)
		
		l:AddPadding(padding)
		
		d.undoButton = LM.GUI.ImageButton('ScriptResources/mr_name/mr_undo', MR_Name:Localize('Undo'), false, d.UNDO, false)
		l:AddChild(d.undoButton, LM.GUI.ALIGN_LEFT, 0)
		
		l:AddPadding(padding)
		
		l:AddChild(LM.GUI.Divider(true), LM.GUI.ALIGN_FILL)
		
		l:AddPadding(padding)
		
		d.rowInput = LM.GUI.TextControl(28, '', self.ROW, LM.GUI.FIELD_INT, MR_Name:Localize('Row:'))
		l:AddChild(d.rowInput, LM.GUI.ALIGN_LEFT, 0)
		
		l:AddPadding(padding)
		
		d.slotSizeInput = LM.GUI.TextControl(35, '', self.SLOT_SIZE, LM.GUI.FIELD_INT, MR_Name:Localize('Slot size:'))
		l:AddChild(d.slotSizeInput, LM.GUI.ALIGN_LEFT, 0)
		
		l:AddPadding(padding)
		
		d.slotsInput = LM.GUI.TextControl(35, '', self.SLOTS, LM.GUI.FIELD_INT, MR_Name:Localize('Slots:'))
		l:AddChild(d.slotsInput, LM.GUI.ALIGN_LEFT, 0)
		
		l:AddPadding(padding)
		
		d.minusSlotButton = LM.GUI.ImageButton('ScriptResources/mr_name/mr_minus', MR_Name:Localize('Remove slot'), false, d.MINUS_SLOT, false)
		l:AddChild(d.minusSlotButton, LM.GUI.ALIGN_LEFT, 0)
		
		l:AddPadding(padding)
		
		d.plusSlotButton = LM.GUI.ImageButton('ScriptResources/mr_name/mr_plus', MR_Name:Localize('Add slot'), false, d.PLUS_SLOT, false)
		l:AddChild(d.plusSlotButton, LM.GUI.ALIGN_LEFT, 0)
		
		l:AddPadding(padding)
		
		l:PushV()
			l:AddPadding(-3)
			
			d.expandsButton = LM.GUI.ImageButton('ScriptResources/mr_name/mr_expand', MR_Name:Localize('Expand/Collapse'), true, d.EXPAND, false)
			l:AddChild(d.expandsButton, LM.GUI.ALIGN_LEFT, 0)
		l:Pop()
		
	l:Pop()
	
	if MR_Name.extraMenu then
		l:AddChild(LM.GUI.Divider(false), LM.GUI.ALIGN_FILL)
		
		l:AddPadding(padding)
		
		l:PushH()
			d.findInput = LM.GUI.TextControl(150, '', self.FIND, LM.GUI.FIELD_TEXT, MR_Name:Localize('Find:'))
			l:AddChild(d.findInput, LM.GUI.ALIGN_LEFT, 0)
			
			l:AddPadding(padding)
			
			d.replaceInput = LM.GUI.TextControl(150, '', self.REPLACE, LM.GUI.FIELD_TEXT, MR_Name:Localize('Replace:'))
			l:AddChild(d.replaceInput, LM.GUI.ALIGN_LEFT, 0)
			
			d.replaceButton = LM.GUI.ImageButton('ScriptResources/mr_name/mr_replace', MR_Name:Localize('Replace'), false, d.REPLACE_BUTTON, false)
			d.replaceButton:SetAlternateMessage(d.REPLACE_BUTTON_ALT)
			l:AddChild(d.replaceButton, LM.GUI.ALIGN_LEFT, 0)
			
			l:AddPadding(padding + 2)
			
			d.selectButton = LM.GUI.ImageButton('ScriptResources/mr_name/mr_select', MR_Name:Localize('Select'), false, d.SELECT_BUTTON, false)
			d.selectButton:SetAlternateMessage(d.SELECT_BUTTON_ALT)
			d.selectButton:SetToolTip(MR_Name:Localize('Select tooltip'))
			l:AddChild(d.selectButton, LM.GUI.ALIGN_LEFT, 0)
			
			l:AddPadding(padding + 2)

			d.showSelectedBonesButton = LM.GUI.ImageButton('ScriptResources/mr_name/mr_visibility', MR_Name:Localize('Show selected bones'), true, d.SHOW_SELECTED_BONES, false)
			l:AddChild(d.showSelectedBonesButton, LM.GUI.ALIGN_LEFT, 0)

			l:AddPadding(padding + 2)

			l:AddChild(LM.GUI.Divider(true), LM.GUI.ALIGN_FILL)

			l:AddPadding(padding + 1)
			
			d.clearPrefixButton = LM.GUI.ImageButton('ScriptResources/mr_name/mr_clear_prefix', MR_Name:Localize('Clear prefix'), false, d.CLEAR_PREFIX, false)
			d.clearPrefixButton:SetAlternateMessage(d.CLEAR_PREFIX_ALT)
			l:AddChild(d.clearPrefixButton, LM.GUI.ALIGN_LEFT, 0)
			
			l:AddPadding(padding)
			
			d.clearSuffixButton = LM.GUI.ImageButton('ScriptResources/mr_name/mr_clear_suffix', MR_Name:Localize('Clear suffix'), false, d.CLEAR_SUFFIX, false)
			d.clearSuffixButton:SetAlternateMessage(d.CLEAR_SUFFIX_ALT)
			l:AddChild(d.clearSuffixButton, LM.GUI.ALIGN_LEFT, 0)
		l:Pop()	
		
		l:AddPadding(padding)
		
	end
		
	d.dummyList = LM.GUI.ImageTextList(0, 1, LM.GUI.MSG_CANCEL)
	d.dummyList:AddItem('', false)
	l:AddChild(d.dummyList, LM.GUI.ALIGN_FILL, 0)
	
	local stringLength = string.rep(" ", MR_Name.slotSize)
	
	l:PushV()
		l:PushH()
			for i=1, MR_Name.slots do
				d.nameButton[i] = LM.GUI.Button(stringLength, self.NAME_BUTTON + i)
				d.nameButton[i]:SetAlternateMessage(self.NAME_BUTTON_ALT + i)
				l:AddChild(d.nameButton[i], LM.GUI.ALIGN_LEFT, 0)
				
				l:AddPadding(-15)
				
				if i % MR_Name.row == 0 then
					l:Pop()
					l:AddPadding(-15)
					l:PushH()
				end
			end
		l:Pop()
	l:Pop()

	return d
end

function MR_NameDialog:UpdateWidgets(moho)
	self.boneButton:SetValue(MR_Name.bone)
	self.layerButton:SetValue(MR_Name.layer)
	MR_Name.slots = math.floor(MR_Name.slots)
	self.rowInput:SetValue(MR_Name.row)
	self.slotSizeInput:SetValue(MR_Name.slotSize)
	self.slotsInput:SetValue(MR_Name.slots)
	self.swapModeButton:SetValue(MR_Name.swapMode)
	self.insertAtStartButton:SetValue(MR_Name.insertAtStart)
	self.namingModeButton:SetValue(MR_Name.namingMode)
	
	self.expandsButton:SetValue(MR_Name.extraMenu)
	if self.findInput ~= nil and MR_Name.extraMenu then
		self.findInput:SetValue(MR_Name.findText)
	end
	
	if self.replaceInput ~= nil and MR_Name.extraMenu then
		self.replaceInput:SetValue(MR_Name.replaceText)
	end

	if self.showSelectedBonesButton ~= nil then
		self.showSelectedBonesButton:SetValue(MR_Name.showSelectedBones)
	end
	
	for i=1, MR_Name.slots do
		if MR_Name['name'..i] == nil then
			MR_Name['name'..i] = ''
		end
	end
	
	for i=1, MR_Name.slots do
		if self.nameButton[i] then
			self.nameButton[i]:SetLabel(MR_Name['name'..i], false)
			self.nameButton[i]:Redraw()
		end
	end
end

function MR_NameDialog:OnOK(moho)
	if MR_Name.needReopen then
		MR_Name:Run(moho)
		MR_Name.needReopen = false
	else
		MR_Name.nameDlog = nil
		MR_Name.needReopen = false
	end
end

function MR_NameDialog:HandleMessage(msg)
	MR_Name.slots = math.floor(MR_Name.slots)
	if msg == LM.GUI.MSG_CANCEL then
		return
	elseif msg == self.BONE then
		self.boneButton:SetValue(true)
		self.layerButton:SetValue(false)
		
		MR_Name.bone = true
		MR_Name.layer = false
	elseif msg == self.LAYER then
		self.boneButton:SetValue(false)
		self.layerButton:SetValue(true)
		
		MR_Name.bone = false
		MR_Name.layer = true
	elseif msg == self.LOAD then
		MR_Name.lastSlotsList = {}
		MR_Name.lastSlots = MR_Name.slots
		for i=1, MR_Name.maxSlots do
			MR_Name.lastSlotsList[i] = MR_Name['name'..i]
		end
		MR_Name:Load()
	elseif msg == self.SAVE then
		MR_Name:Save()
	elseif msg == self.CLEAR_SLOTS then
		MR_Name.lastSlotsList = {}
		MR_Name.lastSlots = MR_Name.slots
		for i=1, MR_Name.maxSlots do
			MR_Name.lastSlotsList[i] = MR_Name['name'..i]
			MR_Name['name'..i] = ''
		end
		self:UpdateWidgets(moho)
	elseif msg == self.SORT then
		MR_Name.lastSlotsList = {}
		MR_Name.lastSlots = MR_Name.slots
		for i=1, MR_Name.maxSlots do
			MR_Name.lastSlotsList[i] = MR_Name['name'..i]
		end
		MR_Name:Sort(true)
	elseif msg == self.SORT_ALT then
		MR_Name.lastSlotsList = {}
		MR_Name.lastSlots = MR_Name.slots
		for i=1, MR_Name.maxSlots do
			MR_Name.lastSlotsList[i] = MR_Name['name'..i]
		end
		MR_Name:Sort(false)
	elseif msg == self.NAMING_MODE then
		MR_Name.namingMode = self.namingModeButton:Value()
		if MR_Name.namingMode then
			MR_Name.swapMode = false
			self.swapModeButton:SetValue(false)
		end
	elseif msg == self.SWAP_MODE then
		MR_Name.swapMode = self.swapModeButton:Value()
		if MR_Name.swapMode then
			MR_Name.namingMode = false
			self.namingModeButton:SetValue(false)
		end
		if not MR_Name.swapMode then
			MR_Name.swapSlotId = 0
		end
	elseif msg == self.UNDO then
		MR_Name.swapSlotId = 0
		if #MR_Name.lastSlotsList > 0 then
			local needRedraw = false
			if MR_Name.slots ~= MR_Name.lastSlots then
				needRedraw = true
			end
			MR_Name.slots = MR_Name.lastSlots
			for i=1, MR_Name.maxSlots do
				if MR_Name.lastSlotsList[i] then
					MR_Name['name'..i] = MR_Name.lastSlotsList[i]
				end
			end
			MR_Name.lastSlotsList = {}
			MR_Name.lastSlots = 0
			if needRedraw then
				self.dummyList:SetSelItem(self.dummyList:GetItem(0), false)
				if MR_Name.nameDlog then
					MR_Name.nameDlog = nil
					MR_Name.needReopen = true
				end
			else
				for i=1, MR_Name.slots do
					local val = MR_Name['name'..i]
					if val then
						self.nameButton[i]:SetLabel(val, false)
						self.nameButton[i]:Redraw()
					end
				end
			end
		end
	elseif msg == self.ROW then
		local minRow = math.ceil(MR_Name.slots / MR_Name.maxRows)
		MR_Name.row = LM.Clamp(self.rowInput:IntValue(), minRow, MR_Name.rowMax)
		MR_Name.row = LM.Clamp(MR_Name.row, MR_Name.rowMin, MR_Name.rowMax)
		self.dummyList:SetSelItem(self.dummyList:GetItem(0), false)
		if MR_Name.nameDlog then
			MR_Name.nameDlog = nil
			MR_Name.needReopen = true
		end
	elseif msg == self.SLOT_SIZE then
		MR_Name.slotSize = LM.Clamp(self.slotSizeInput:IntValue(), MR_Name.slotSizeMin, MR_Name.slotSizeMax)
		self.dummyList:SetSelItem(self.dummyList:GetItem(0), false)
		if MR_Name.nameDlog then
			MR_Name.nameDlog = nil
			MR_Name.needReopen = true
		end
	elseif msg == self.SLOTS then
		MR_Name.lastSlotsList = {}
		MR_Name.lastSlots = MR_Name.slots
		for i=1, MR_Name.maxSlots do
			MR_Name.lastSlotsList[i] = MR_Name['name'..i]
		end
		
		MR_Name.slots = LM.Clamp(self.slotsInput:Value(), 1, MR_Name.maxSlots)
		MR_Name.slots = math.floor(MR_Name.slots)
		
		local minRow = math.ceil(MR_Name.slots / MR_Name.maxRows)
		MR_Name.row = LM.Clamp(MR_Name.row, minRow, MR_Name.rowMax)
		
		self.dummyList:SetSelItem(self.dummyList:GetItem(0), false)
		if MR_Name.nameDlog then
			MR_Name.nameDlog = nil
			MR_Name.needReopen = true
		end
	elseif msg == self.MINUS_SLOT then
		MR_Name.lastSlotsList = {}
		MR_Name.lastSlots = MR_Name.slots
		for i=1, MR_Name.maxSlots do
			MR_Name.lastSlotsList[i] = MR_Name['name'..i]
		end
		
		MR_Name.slots = LM.Clamp(MR_Name.slots - 1, 1, MR_Name.maxSlots)
		MR_Name.slots = math.floor(MR_Name.slots)
		
		local minRow = math.ceil(MR_Name.slots / MR_Name.maxRows)
		MR_Name.row = LM.Clamp(MR_Name.row, minRow, MR_Name.rowMax)
		
		self.dummyList:SetSelItem(self.dummyList:GetItem(0), false)
		if MR_Name.nameDlog then
			MR_Name.nameDlog = nil
			MR_Name.needReopen = true
		end
	elseif msg == self.PLUS_SLOT then
		MR_Name.lastSlotsList = {}
		MR_Name.lastSlots = MR_Name.slots
		for i=1, MR_Name.maxSlots do
			MR_Name.lastSlotsList[i] = MR_Name['name'..i]
		end
		
		MR_Name.slots = LM.Clamp(MR_Name.slots + 1, 1, MR_Name.maxSlots)
		MR_Name.slots = math.floor(MR_Name.slots)
		
		local minRow = math.ceil(MR_Name.slots / MR_Name.maxRows)
		MR_Name.row = LM.Clamp(MR_Name.row, minRow, MR_Name.rowMax)
		
		self.dummyList:SetSelItem(self.dummyList:GetItem(0), false)
		if MR_Name.nameDlog then
			MR_Name.nameDlog = nil
			MR_Name.needReopen = true
		end
	elseif msg >= self.NAME_BUTTON and msg <= self.NAME_BUTTON + MR_Name.slots then
		if MR_Name.namingMode then
			MR_Name.lastSlotsList = {}
			MR_Name.lastSlots = MR_Name.slots
			for i=1, MR_Name.maxSlots do
				MR_Name.lastSlotsList[i] = MR_Name['name'..i]
			end
			
			MR_Name.index = math.floor(msg - self.NAME_BUTTON)
			
			local dlog = MR_NameSetTextDialog:new(moho)
			if (dlog:DoModal() == LM.GUI.MSG_CANCEL) then
				return
			end
			self.nameButton[MR_Name.index]:SetLabel(MR_Name['name'..MR_Name.index], false)
			self.nameButton[MR_Name.index]:Redraw()
			MR_Name.index = nil
			return
		end
		local index = math.floor(msg - self.NAME_BUTTON)
		if MR_Name.swapMode then
			if MR_Name.swapSlotId == 0 then
				MR_Name.swapSlotId = index
				return
			elseif MR_Name.swapSlotId > 0 then	
				MR_Name.lastSlotsList = {}
				MR_Name.lastSlots = MR_Name.slots
				for i=1, MR_Name.maxSlots do
					MR_Name.lastSlotsList[i] = MR_Name['name'..i]
				end
				MR_Name:SwapSlots(MR_Name.swapSlotId, index)
				
				self.nameButton[MR_Name.swapSlotId]:SetLabel(MR_Name['name'..MR_Name.swapSlotId], false)
				self.nameButton[MR_Name.swapSlotId]:Redraw()
				
				self.nameButton[index]:SetLabel(MR_Name['name'..index], false)
				self.nameButton[index]:Redraw()
				
				MR_Name.swapSlotId = 0
				return
			end
		end
		local text = MR_Name['name'..index]
		if MR_Name.bone then
			local helper = MOHO.ScriptInterfaceHelper:new_local()
			local moho = helper:MohoObject()
			if moho.document == nil then
				helper:delete()
				return
			end
			local skel = moho:Skeleton()
			if skel == nil then
				helper:delete()
				return
			end
			if moho:CountSelectedBones() > 0 then
				moho.document:SetDirty()
				moho.document:PrepUndo(moho.layer, true)
				for i=0, skel:CountBones()-1 do
					local bone = skel:Bone(i)
					if bone.fSelected then
						if MR_Name.insertAtStart then
							bone:SetName(text..bone:Name())
						else
							bone:SetName(bone:Name()..text)
						end
					end
				end
			end
		
			moho:UpdateUI()
			moho.view:DrawMe()
			helper:delete()	
		elseif MR_Name.layer then
			local helper = MOHO.ScriptInterfaceHelper:new_local()
			local moho = helper:MohoObject()
			
			if moho.document == nil then
				helper:delete()
				return
			end
			local selectedLayers = moho.document:CountSelectedLayers()
			if selectedLayers == 1 then
				local layer = moho.layer
				moho.document:SetDirty()
				moho.document:PrepUndo(layer, true)
				if MR_Name.insertAtStart then
					layer:SetName(text..layer:Name())
				else
					layer:SetName(layer:Name()..text)
				end
			elseif selectedLayers > 1 then
				moho.document:SetDirty()
				moho.document:PrepUndo(nil)
				for i=0, selectedLayers -1 do
					local layer = moho.document:GetSelectedLayer(i)
					if MR_Name.insertAtStart then
						layer:SetName(text..layer:Name())
					else
						layer:SetName(layer:Name()..text)
					end
				end
			end
			
			moho:UpdateUI()
			helper:delete()	
		end
	elseif msg >= self.NAME_BUTTON_ALT and msg <= self.NAME_BUTTON_ALT + MR_Name.slots then
		if MR_Name.swapMode then
			return
		end

		local index = math.floor(msg - self.NAME_BUTTON_ALT)
		local text = MR_Name['name'..index]

		if MR_Name.namingMode and MR_Name.extraMenu then
			if self.findInput ~= nil then
				MR_Name.findText = text
				self.findInput:SetValue(MR_Name.findText)
			end

			return
		end
		if MR_Name.bone then
			local helper = MOHO.ScriptInterfaceHelper:new_local()
			local moho = helper:MohoObject()
			if moho.document == nil then
				helper:delete()
				return
			end
			local skel = moho:Skeleton()
			if skel == nil then
				helper:delete()
				return
			end
			if moho:CountSelectedBones() > 0 then
				moho.document:SetDirty()
				moho.document:PrepUndo(moho.layer, true)
				for i=0, skel:CountBones()-1 do
					local bone = skel:Bone(i)
					if bone.fSelected then
						bone:SetName(text)
					end
				end
			end
		
			moho:UpdateUI()
			moho.view:DrawMe()
			helper:delete()	
		elseif MR_Name.layer then
			local helper = MOHO.ScriptInterfaceHelper:new_local()
			local moho = helper:MohoObject()
			
			if moho.document == nil then
				helper:delete()
				return
			end
			local selectedLayers = moho.document:CountSelectedLayers()
			if selectedLayers == 1 then
				local layer = moho.layer
				moho.document:SetDirty()
				moho.document:PrepUndo(layer, true)
				layer:SetName(text)
			elseif selectedLayers > 1 then
				moho.document:SetDirty()
				moho.document:PrepUndo(nil)
				for i=0, selectedLayers -1 do
					local layer = moho.document:GetSelectedLayer(i)
					layer:SetName(text)
				end
			end
			
			moho:UpdateUI()
			helper:delete()	
		end
	elseif msg == self.EXPAND then
		MR_Name.extraMenu = self.expandsButton:Value()
		self.dummyList:SetSelItem(self.dummyList:GetItem(0), false)
		if MR_Name.nameDlog then
			MR_Name.nameDlog = nil
			MR_Name.needReopen = true
		end
	elseif msg == self.FIND then
		MR_Name.findText = self.findInput:Value()
	elseif msg == self.REPLACE then
		MR_Name.replaceText = self.replaceInput:Value()
	elseif msg == self.REPLACE_BUTTON then
		local helper = MOHO.ScriptInterfaceHelper:new_local()
		local moho = helper:MohoObject()
		if moho.document == nil then
			helper:delete()
			return
		end
		MR_Name:ReplaceText(moho, false)
		helper:delete()
	elseif msg == self.REPLACE_BUTTON_ALT then
		local helper = MOHO.ScriptInterfaceHelper:new_local()
		local moho = helper:MohoObject()
		if moho.document == nil then
			helper:delete()
			return
		end
		MR_Name:ReplaceText(moho, true)
		helper:delete()
	elseif msg == self.SELECT_BUTTON then
		local helper = MOHO.ScriptInterfaceHelper:new_local()
		local moho = helper:MohoObject()
		if moho.document == nil then
			helper:delete()
			return
		end
		if MR_Name.bone then
			MR_Name:SelectBones(moho)
		elseif MR_Name.layer then
			MR_Name:SelectLayers(moho)
		end
		helper:delete()
	elseif msg == self.SELECT_BUTTON_ALT then
		local helper = MOHO.ScriptInterfaceHelper:new_local()
		local moho = helper:MohoObject()
		if moho.document == nil then
			helper:delete()
			return
		end
		if MR_Name.bone then
			MR_Name:SelectBones(moho, true)
		elseif MR_Name.layer then
			MR_Name:SelectLayers(moho, true)
		end
		helper:delete()
	elseif msg == self.SHOW_SELECTED_BONES then
		MR_Name.showSelectedBones = self.showSelectedBonesButton:Value()
	elseif msg == self.CLEAR_PREFIX then
		local helper = MOHO.ScriptInterfaceHelper:new_local()
		local moho = helper:MohoObject()
		if moho.document == nil then
			helper:delete()
			return
		end
		MR_Name:ClearPrefixOrSuffix(moho, true, false)
		helper:delete()
	elseif msg == self.CLEAR_PREFIX_ALT then
		local helper = MOHO.ScriptInterfaceHelper:new_local()
		local moho = helper:MohoObject()
		if moho.document == nil then
			helper:delete()
			return
		end
		MR_Name:ClearPrefixOrSuffix(moho, true, true)
		helper:delete()
	elseif msg == self.CLEAR_SUFFIX then
		local helper = MOHO.ScriptInterfaceHelper:new_local()
		local moho = helper:MohoObject()
		if moho.document == nil then
			helper:delete()
			return
		end
		MR_Name:ClearPrefixOrSuffix(moho, false, false)
		helper:delete()	
	elseif msg == self.CLEAR_SUFFIX_ALT then
		local helper = MOHO.ScriptInterfaceHelper:new_local()
		local moho = helper:MohoObject()
		if moho.document == nil then
			helper:delete()
			return
		end
		MR_Name:ClearPrefixOrSuffix(moho, false, true)
		helper:delete()
	elseif msg == self.INSERT_AT_START then
		MR_Name.insertAtStart = self.insertAtStartButton:Value()
	end
end

-- **************************************************
-- The guts of this script
-- **************************************************

function MR_Name:Run(moho)
	if self.nameDlog == nil then
		self.nameDlog = MR_NameDialog:new(moho)
		if self.nameDlog:DoModeless() == LM.GUI.MSG_CANCEL then
			return
		end
	end
end

function MR_Name:Load()
	if not MR_Name:VerifyUtilities() then
		return
	end
	
	local fileName = LM.GUI.OpenFile(MR_Name:Localize("Load slots"))
	if fileName ~= "" then
		local file = io.open(fileName, "r")
		if file then
			local content = file:read("*all")

			file:close()
			local namesList = {}

			local names = ''
			local row = -1
			local slotSize = -1

			for line in content:gmatch("[^\r\n]+") do
				if names == '' then
					names = line
				elseif row == -1 then
					row = math.floor(tonumber(line) or 0)
				elseif slotSize == -1 then
					slotSize = math.floor(tonumber(line) or 0)
				end
			end

			MR_Utilities:StringToTable(names, namesList, 's')
			if #namesList > 0 then
				MR_Name.slots = math.floor(#namesList)
				for i=1, MR_Name.slots do
					MR_Name['name'..i] = namesList[i]
				end
				MR_Name.nameDlog.dummyList:SetSelItem(MR_Name.nameDlog.dummyList:GetItem(0), false)
				if MR_Name.nameDlog then
					MR_Name.nameDlog = nil
					MR_Name.needReopen = true
				end
			else
				local alertText = MR_Name:Localize('Unable to open file')
				local alert = LM.GUI.Alert(LM.GUI.ALERT_WARNING, alertText)
			end
			if row ~= -1 then
				MR_Name.row = row
			end
			if slotSize ~= -1 then
				MR_Name.slotSize = slotSize
			end
		end
	end
end

function MR_Name:Save()
	if not MR_Name:VerifyUtilities() then
		return
	end
	
	local namesList = {}
	for i=1, MR_Name.slots do
		namesList[i] = MR_Name['name'..i]
	end
	-- names, row, slotSize
	local text = MR_Utilities:TableToString(namesList)..'\n'..math.floor(MR_Name.row)..'\n'..math.floor(MR_Name.slotSize)	
	local fileName = LM.GUI.SaveFile(MR_Name:Localize("Save slots"))
	if fileName ~= "" then
		if not fileName:match("%.[^%.]+$") then
			fileName = fileName .. ".nms"
		end
		local f = io.open(fileName, "w")
		f:write(text)
		f:close()
	end
end

function MR_Name:VerifyUtilities()
	if not MR_Utilities then
		local alertText = MR_Name:Localize('The file mr_utilities.lua is missing')
		local alert = LM.GUI.Alert(LM.GUI.ALERT_WARNING, alertText)
		return false
	end
	local utilitiesVersion = tonumber(MR_Utilities:Version())
	if utilitiesVersion < 1.4 then
		local alertText = 'The version of mr_utilities.lua is '..utilitiesVersion..'. Version 1.4 or higher required.'
		local alert = LM.GUI.Alert(LM.GUI.ALERT_WARNING, alertText)
		return false
	end
	return true
end

function MR_Name:Sort(mode)
	if not MR_Name:VerifyUtilities() then
		return
	end
	
	local namesList = {}
	for i=1, MR_Name.slots do
		namesList[i] = MR_Name['name'..i]
	end
	MR_Utilities:SortTableAlphabetically(namesList, mode)
	for i=1, MR_Name.slots do
		MR_Name['name'..i] = namesList[i]
	end
	for i=1, MR_Name.slots do
		local val = MR_Name['name'..i]
		if val then
			MR_Name.nameDlog.nameButton[i]:SetLabel(val, false)
			MR_Name.nameDlog.nameButton[i]:Redraw()
		end
	end
end

function MR_Name:ReplaceText(moho, inner)
	if not MR_Name:VerifyUtilities() then
		return
	end
	
	if MR_Name.findText == '' or MR_Name.findText == nil then
		return
	end
	
	if MR_Name.bone then
		local skel = moho:Skeleton()
		if skel == nil then
			return
		end
		if moho:CountSelectedBones() > 0 then
			moho.document:SetDirty()
			moho.document:PrepUndo(moho.layer, true)
			for i=0, skel:CountBones()-1 do
				local bone = skel:Bone(i)
				if bone.fSelected then
					local name = bone:Name()
					local newName = MR_Utilities:ReplaceTextInString(name, MR_Name.findText, MR_Name.replaceText)
					bone:SetName(newName)
				end
			end
		end
		moho:UpdateUI()
		moho.view:DrawMe()
		moho.layer:UpdateCurFrame(true)
	elseif MR_Name.layer then
		if inner then
			local count = 0
			local layersToReplace = {}
			repeat
				local layer = moho.document:LayerByAbsoluteID(count)
				if layer then
					if layer:IsAncestorSelected() or layer == moho.layer or layer:SecondarySelection() then
						table.insert(layersToReplace, count)
					end
					count = count + 1
				end
			until not layer
			if #layersToReplace == 1 then
				local layer = moho.document:LayerByAbsoluteID(layersToReplace[1])
				moho.document:SetDirty()
				moho.document:PrepUndo(layer, true)
			elseif #layersToReplace > 1 then
				moho.document:SetDirty()
				moho.document:PrepUndo(nil)
			end

			for i=1, #layersToReplace do
				local layer = moho.document:LayerByAbsoluteID(layersToReplace[i])
				local name = layer:Name()
				local newName = MR_Utilities:ReplaceTextInString(name, MR_Name.findText, MR_Name.replaceText)
				layer:SetName(newName)
			end
		else
			local selectedLayers = moho.document:CountSelectedLayers()
			if selectedLayers == 1 then
				local layer = moho.layer
				moho.document:SetDirty()
				moho.document:PrepUndo(layer, true)
				local name = layer:Name()
				local newName = MR_Utilities:ReplaceTextInString(name, MR_Name.findText, MR_Name.replaceText)
				layer:SetName(newName)
			elseif selectedLayers > 1 then
				moho.document:SetDirty()
				moho.document:PrepUndo(nil)
				for i=0, selectedLayers -1 do
					local layer = moho.document:GetSelectedLayer(i)
					local name = layer:Name()
					local newName = MR_Utilities:ReplaceTextInString(name, MR_Name.findText, MR_Name.replaceText)
					layer:SetName(newName)
				end
			end
		end
		moho:UpdateUI()
	end
end

function MR_Name:ClearPrefixOrSuffix(moho, prefix, alt)
	if not MR_Name:VerifyUtilities() then
		return
	end
	
	if MR_Name.bone then
		local skel = moho:Skeleton()
		if skel == nil then
			return
		end
		if moho:CountSelectedBones() > 0 then
			moho.document:SetDirty()
			moho.document:PrepUndo(moho.layer, true)
			for i=0, skel:CountBones()-1 do
				local bone = skel:Bone(i)
				if bone.fSelected then
					local name = bone:Name()
					local newName = ""
					if prefix then
						newName = name:sub(2)
					else
						newName = name:sub(1, -2)
					end
					bone:SetName(newName)
				end
			end
		end
		moho:UpdateUI()
		moho.view:DrawMe()
		moho.layer:UpdateCurFrame(true)
	elseif MR_Name.layer then
		if alt then
			local count = 0
			local layersToRename = {}
			repeat
				local layer = moho.document:LayerByAbsoluteID(count)
				if layer then
					if layer:IsAncestorSelected() or layer == moho.layer or layer:SecondarySelection() then
						table.insert(layersToRename, count)
					end
					count = count + 1
				end
			until not layer
			if #layersToRename == 1 then
				local layer = moho.document:LayerByAbsoluteID(layersToRename[1])
				moho.document:SetDirty()
				moho.document:PrepUndo(layer, true)
			elseif #layersToRename > 1 then
				moho.document:SetDirty()
				moho.document:PrepUndo(nil)
			end

			for i=1, #layersToRename do
				local layer = moho.document:LayerByAbsoluteID(layersToRename[i])
				local name = layer:Name()
				local newName = ""
				if prefix then
					newName = name:sub(2)
				else
					newName = name:sub(1, -2)
				end
				layer:SetName(newName)
			end
		else
			local selectedLayers = moho.document:CountSelectedLayers()
			if selectedLayers == 1 then
				local layer = moho.layer
				moho.document:SetDirty()
				moho.document:PrepUndo(layer, true)
				local name = layer:Name()
				local newName = ""
				if prefix then
					newName = name:sub(2)
				else
					newName = name:sub(1, -2)
				end
				layer:SetName(newName)
			elseif selectedLayers > 1 then
				moho.document:SetDirty()
				moho.document:PrepUndo(nil)
				for i=0, selectedLayers -1 do
					local layer = moho.document:GetSelectedLayer(i)
					local name = layer:Name()
					local newName = ""
					if prefix then
						newName = name:sub(2)
					else
						newName = name:sub(1, -2)
					end
					layer:SetName(newName)
				end
			end
		end
		moho:UpdateUI()
	end
end

function MR_Name:SwapSlots(slot1, slot2)
	local slot1Val = MR_Name['name'..slot1]
	local slot2Val = MR_Name['name'..slot2]
	if not slot1Val or not slot2Val then
		MR_Name.swapSlotId = 0
		return
	end
	MR_Name['name'..slot1] = slot2Val
	MR_Name['name'..slot2] = slot1Val
end

function MR_Name:SelectBones(moho, matchCase)
	local skel = moho:Skeleton()
	if skel == nil then
		return
	end

	local bonesToSelect = {}
	for i=0, skel:CountBones()-1 do
		local bone = skel:Bone(i)
		local name = bone:Name()
		if not matchCase then
			name = name:lower()
			if name:find(MR_Name.findText:lower()) then
				table.insert(bonesToSelect, i)
			end
		else
			if name:find(MR_Name.findText) then
				table.insert(bonesToSelect, i)
			end
		end
	end
	if #bonesToSelect > 0 then
		skel:SelectNone()
		for i=1, #bonesToSelect do
			local bone = skel:Bone(bonesToSelect[i])
			bone.fSelected = true
			if MR_Name.showSelectedBones then
				bone.fHidden = false
			end
		end
		moho:UpdateUI()
		moho.view:DrawMe()
	end
end

function MR_Name:SelectLayers(moho, matchCase)
	local layersToSelect = {}
	local count = 0
	repeat
		local layer = moho.document:LayerByAbsoluteID(count)
		if layer then
			local name = layer:Name()
			if not matchCase then
				local cleanName = name:lower():gsub("^%s*(.-)%s*$", "%1")
				local cleanFindText = MR_Name.findText:lower():gsub("^%s*(.-)%s*$", "%1")

				if cleanName:find(cleanFindText) then
					table.insert(layersToSelect, count)
				end
			else
				local cleanName = name:gsub("^%s*(.-)%s*$", "%1")
				local cleanFindText = MR_Name.findText:gsub("^%s*(.-)%s*$", "%1")

				if cleanName:find(cleanFindText) then
					table.insert(layersToSelect, count)
				end
			end
			count = count + 1
		end
	until not layer
	if #layersToSelect > 0 then
		local firstSelection = true
		for i=1, #layersToSelect do
			local layer = moho.document:LayerByAbsoluteID(layersToSelect[i])
			if firstSelection then
				moho:SetSelLayer(layer)
				firstSelection = false
			else
				moho:SetSelLayer(layer, true)
			end
			moho:ShowLayerInLayersPalette(layer)
		end
		moho:UpdateUI()
		moho.view:DrawMe()
	end
end

-- **************************************************
-- Localization
-- **************************************************

function MR_Name:Localize(text)
	local phrase = {}

	phrase['Description'] = 'Easily label bones and layers using templates. Alt + click on the name button to fully overwrite the name.'
	phrase['UILabel'] = 'MR Name '..self:Version()

	phrase['Bone'] = 'Bone'
	phrase['Layer'] = 'Layer'
	phrase['Save slots'] = 'Save slots'
	phrase['Load slots'] = 'Load slots'
	phrase['Slots:'] = 'Slots:'
	phrase['Naming mode'] = 'Naming mode'
	phrase['Clear slots'] = 'Clear slots'
	phrase['Sort alphabetically'] = 'Sort alphabetically (Alt + click to sort including all symbols)'
	phrase['Swap mode'] = 'Swap slots mode'
	phrase['Undo'] = 'Undo the last slot changes'
	phrase['Row:'] = 'Row:'
	phrase['Slot size:'] = 'Size:'
	phrase['Load'] = 'Load'
	phrase['Save'] = 'Save'
	
	phrase['Enter text'] = 'Enter text'
	phrase['Remove slot'] = 'Remove slot'
	phrase['Add slot'] = 'Add slot'
	phrase['Expand/Collapse'] = 'Expand/Collapse'
	phrase['Find:'] = 'Find:'
	phrase['Replace:'] = 'Replace:'
	phrase['Replace'] = 'Replace text (Alt + click to apply to inner layers)'
	phrase['Select'] = 'Select'
	phrase['Select tooltip'] = 'Select bones or layers containing the text (Alt + click to match case)'
	phrase['Show selected bones'] = 'Show selected bones'
	phrase['Clear prefix'] = 'Clear first character (Alt + click to apply to inner layers)'
	phrase['Clear suffix'] = 'Clear last character (Alt + click to apply to inner layers)'
	phrase['Insert text at start'] = 'Insert text at start'
	
	phrase['The file mr_utilities.lua is missing'] = 'The file mr_utilities.lua is missing'
	phrase['The file mr_utilities.lua is missing'] = 'The file mr_utilities.lua is missing'
	phrase['Unable to open file'] = 'Unable to open file'

	return phrase[text]
end