-- **************************************************
-- Provide Moho with the name of this script object
-- **************************************************

ScriptName = "SZ_ZTranslateWithAutoScale"

-- **************************************************
-- General information about this script
-- **************************************************

SZ_ZTranslateWithAutoScale = {}

function SZ_ZTranslateWithAutoScale:Name()
	return "Z-Translate with auto-scale"
end

function SZ_ZTranslateWithAutoScale:Version()
	return "1.0"
end

function SZ_ZTranslateWithAutoScale:Description()
	return "Drag to move this layer along the Z axis with automatic proportional scale"
end

function SZ_ZTranslateWithAutoScale:Creator()
	return "Stan from 2danimator.ru"
end

function SZ_ZTranslateWithAutoScale:UILabel()
	return "Z-Translate with auto scale"
end

-- **************************************************
-- Recurring values
-- **************************************************
SZ_ZTranslateWithAutoScale.allowBehind = false
SZ_ZTranslateWithAutoScale.savedVal = nil
SZ_ZTranslateWithAutoScale.camPos = nil
SZ_ZTranslateWithAutoScale.layerPos = nil
SZ_ZTranslateWithAutoScale.layerScale = nil

-- **************************************************
-- The guts of this script
-- **************************************************

function SZ_ZTranslateWithAutoScale:IsEnabled(moho)
	if (moho.layerFrame > 0) then
		return true
	end
	return false
end


function SZ_ZTranslateWithAutoScale:IsRelevant(moho)
	return true
end


function SZ_ZTranslateWithAutoScale:OnMouseDown(moho, mouseEvent)
	local layer = moho.layer
	moho.document:PrepUndo(layer)
	moho.document:SetDirty()
	
	self.savedVal = self.zValue:FloatValue()
	self.camPos = moho.document.fCameraTrack:GetValue(0)
	self.layerPos = moho.layer.fTranslation:GetValue(0)
	self.layerScale = moho.layer.fScale:GetValue(0)
	
	local newVector2 = LM.Vector2:new_local()
	
	newVector2:Set(self.camPos.x - self.layerPos.x, self.camPos.y - self.layerPos.y)
	newVector2 = newVector2 / self.layerScale.z
	layer:SetOriginWithTransformCorrection(newVector2)	
	
	moho.layer:UpdateCurFrame()
	mouseEvent.view:DrawMe()
end

function SZ_ZTranslateWithAutoScale:OnMouseMoved(moho, mouseEvent)
	local eventVal = (mouseEvent.pt.y - mouseEvent.startPt.y) / 4
	local newVal = self.savedVal + eventVal
	
	self:ChangeZCoord(moho, newVal)
	
	moho.layer:UpdateCurFrame()
	mouseEvent.view:DrawMe()
end

function SZ_ZTranslateWithAutoScale:OnMouseUp(moho, mouseEvent)
	local layer = moho.layer
	local vector2 = LM.Vector2:new_local()
	vector2:Set(0, 0)
	layer:SetOriginWithTransformCorrection(vector2)
	
	self.savedVal = nil
	self.camPos = nil
	self.layerPos = nil
	self.layerScale = nil
	moho:NewKeyframe(CHANNEL_LAYER_T)
	moho:NewKeyframe(CHANNEL_LAYER_S)
end

function SZ_ZTranslateWithAutoScale:OnKeyDown(moho, keyEvent)
	LM_SelectPoints:OnKeyDown(moho, keyEvent)
end

-- **************************************************
-- Tool options - create and respond to tool's UI
-- **************************************************
SZ_ZTranslateWithAutoScale.ZVALUE = MOHO.MSG_BASE
SZ_ZTranslateWithAutoScale.RESET = MOHO.MSG_BASE+1
SZ_ZTranslateWithAutoScale.ALLOW_BEHIND = MOHO.MSG_BASE+2

function SZ_ZTranslateWithAutoScale:DoLayout(moho, layout)
	layout:AddChild(LM.GUI.StaticText("Z value:"))
	self.zValue = LM.GUI.TextControl(0, "-00.0000", self.ZVALUE, LM.GUI.FIELD_FLOAT)
	layout:AddChild(self.zValue)
	
	layout:AddChild(LM.GUI.StaticText("Scale:"))
	self.scale = LM.GUI.DynamicText("000.00")
	layout:AddChild(self.scale)
	
	self.reset = LM.GUI.Button("Reset", self.RESET)
	layout:AddChild(self.reset)
	
	layout:AddPadding()
	layout:AddChild(LM.GUI.Divider(true))
	layout:AddPadding()
	
	self.allowBehindCB = LM.GUI.CheckBox("Allow layer to move behind camera", self.ALLOW_BEHIND)
	layout:AddChild(self.allowBehindCB)
end

function SZ_ZTranslateWithAutoScale:UpdateWidgets(moho)
	local layerTrans = moho.layer.fTranslation:GetValue(moho.layerFrame)
	local layerScale = moho.layer.fScale:GetValue(moho.layerFrame)
	self.zValue:SetValue(layerTrans.z)
	self.scale:SetValue(layerScale.z)
	self.allowBehindCB:SetValue(self.allowBehind)
end

function SZ_ZTranslateWithAutoScale:HandleMessage(moho, view, msg)
	if msg == self.ZVALUE then
		moho.document:PrepUndo(moho.layer)
		moho.document:SetDirty()
		local newValue = self.zValue:FloatValue()
		self.savedVal = self.zValue:FloatValue()
		self.camPos = moho.document.fCameraTrack:GetValue(0)
		self.layerPos = moho.layer.fTranslation:GetValue(0)
		self.layerScale = moho.layer.fScale:GetValue(0)
		local newVector2 = LM.Vector2:new_local()	
		newVector2:Set(self.camPos.x - self.layerPos.x, self.camPos.y - self.layerPos.y)
		moho.layer:SetOriginWithTransformCorrection(newVector2)
		
		self:ChangeZCoord(moho, newValue)
		
		newVector2 = LM.Vector2:new_local()
		newVector2:Set(0, 0)
		moho.layer:SetOriginWithTransformCorrection(newVector2)
		self.savedVal = nil
		self.camPos = nil
		self.layerPos = nil
		self.layerScale = nil
		moho:NewKeyframe(CHANNEL_LAYER_T)
		moho:NewKeyframe(CHANNEL_LAYER_S)
	elseif msg == self.RESET then
		local layerPos = moho.layer.fTranslation:GetValue(0)
		local layerScale = moho.layer.fScale:GetValue(0)
		moho.layer.fTranslation:SetValue(moho.layerFrame, layerPos)
		moho.layer.fScale:SetValue(moho.layerFrame, layerScale)
		self.zValue:SetValue(layerPos.z)
		self.scale:SetValue(layerScale.z)
		moho:NewKeyframe(CHANNEL_LAYER_T)
		moho:NewKeyframe(CHANNEL_LAYER_S)
	elseif msg == self.ALLOW_BEHIND then
		self.allowBehind = self.allowBehindCB:Value()
	end
end

function SZ_ZTranslateWithAutoScale:ChangeZCoord(moho, zCoord)
	if self.allowBehind == false then
		if zCoord > (self.camPos.z-.0001) then
			return false
		end
	end
	
	local q = 0
	
	if zCoord <= 0 and self.camPos.z >= 0 then
		q = (math.abs(zCoord) + self.camPos.z) / self.camPos.z
	elseif zCoord >= 0 and self.camPos.z >= 0 then
		q = math.abs(self.camPos.z - zCoord) / self.camPos.z
	elseif zCoord >= 0 and self.camPos.z <= 0 then
		q = math.abs(self.camPos.z) + zCoord / self.camPos.z
	else
		q = math.abs(math.abs(self.camPos.z) - math.abs(zCoord)) / self.camPos.z
	end

	
	self.zValue:SetValue(zCoord)

	self.layerPos:Set(self.layerPos.x, self.layerPos.y, zCoord)
	moho.layer.fTranslation:SetValue(moho.layerFrame, self.layerPos)
	
	local vec3 = LM.Vector3:new_local()
	local newScale = self.layerScale.z * q
	
	vec3:Set(newScale, newScale, newScale)
	moho.layer.fScale:SetValue(moho.layerFrame, vec3)
	
	self.scale:SetValue(newScale)	
end
