MR_Utilities = {}

function MR_Utilities:Version()
	return "1.7"
end

function MR_Utilities:GetDistance(Pos1, Pos2)
	return math.sqrt((Pos2.x-Pos1.x)^2+(Pos2.y-Pos1.y)^2)
end

function MR_Utilities:RotateVector2(vector2, center, angle)
	local centerVec = LM.Vector2:new_local()
	local newVector2 = LM.Vector2:new_local()
	local dif = LM.Vector2:new_local()
	centerVec:Set(center)
	dif:Set(vector2 - centerVec)
	local px = dif.x * math.cos(angle) - dif.y * math.sin(angle)
	local py = dif.x * math.sin(angle) + dif.y * math.cos(angle)
	newVector2:Set(px + centerVec.x, py + centerVec.y)
	return newVector2
end

function MR_Utilities:IsEqual(a, b, epsilon)
    local absA = math.abs(a)
    local absB = math.abs(b)
    local diff = math.abs(a - b)

    if a == b then 
        return true 
    elseif diff < epsilon then 
        return true
    else 
        return false
    end
end

function MR_Utilities:Round(x, n)
	n = 10 ^ (n or 3)
	x = x * n
	if x >= 0 then x = math.floor(x + 0.5) else x = math.ceil(x - 0.5) end
	return x / n
end

function MR_Utilities:GetGlobalPos(moho, layer, pos, camera, frame)
	local frame = frame or moho.frame
	local globalPos = LM.Vector2:new_local()
	globalPos:Set(pos)
	local layerMatrix = LM.Matrix:new_local()
	if camera then
		layer:GetFullTransform(frame, layerMatrix, moho.document)
	else
		layer:GetFullTransform(frame, layerMatrix, nil)
	end
	layerMatrix:Transform(globalPos)
	return globalPos
end

function MR_Utilities:GetLocalPos(moho, layer, globalPos, camera, frame)
	local frame = frame or moho.frame
	local localPos = LM.Vector2:new_local()
	localPos:Set(globalPos)
	local selLayer = layer
	local selLayerMatrix = LM.Matrix:new_local()
	if camera then	
		selLayer:GetFullTransform(frame, selLayerMatrix, moho.document)
	else
		selLayer:GetFullTransform(frame, selLayerMatrix, nil)
	end
	selLayerMatrix:Invert()
	selLayerMatrix:Transform(localPos)
	return localPos
end

function MR_Utilities:GetLayerByUUID(moho, uuid)
	local count = 0
	repeat
		local layer = moho.document:LayerByAbsoluteID(count)
		if layer then
			count = count + 1
			if layer:UUID() == uuid then
				return layer
			end	
		end
	until not layer
	return nil
end

function MR_Utilities:ValueExists(tbl, value)
    for key, val in pairs(tbl) do
        if val == value then
            return key
        end
    end
    return false
end

function MR_Utilities:ReturnToAction(moho, action, layer)
	if action ~= '' then
		local parentGroup = layer:Parent()
		local skelLayer = nil
		
		if layer:LayerType() == MOHO.LT_BONE and layer:HasAction(action) then
			skelLayer = layer
		elseif parentGroup ~= nil then
			local targetGroup = parentGroup
			repeat
				if targetGroup:LayerType() == MOHO.LT_BONE then
					if targetGroup:HasAction(action) then
						skelLayer = targetGroup
					end	
				end
				targetGroup = targetGroup:Parent()
			until targetGroup == nil
		end
		
		moho:SetSelLayer(skelLayer)
		moho.document:SetCurrentDocAction(action)
		skelLayer:ActivateAction(action)
		if skelLayer ~= layer then
			moho:SetSelLayer(layer)
			layer:ActivateAction(action)
		end
	else	
		moho:SetSelLayer(layer)
	end
end

function MR_Utilities:FindSkeletonLayer(layer)
	local parentGroup = layer:Parent()
	local skelLayer = nil
	
	if layer:LayerType() == MOHO.LT_BONE then
		skelLayer = layer
	elseif parentGroup ~= nil then
		local targetGroup = parentGroup
		repeat
			if targetGroup:LayerType() == MOHO.LT_BONE then
				skelLayer = targetGroup
				return skelLayer
			end
			targetGroup = targetGroup:Parent()
		until targetGroup == nil
	end
	return skelLayer
end

function MR_Utilities:SwapTwoArrayKeys(array, key1, key2)
	local keyTmp1 = array[key1]
	local keyTmp2 = array[key2]
	return keyTmp2, keyTmp1
end

function MR_Utilities:GetMohoVersion(moho)
	local numVers = {}
	local vers = moho:AppVersion()
	for n in string.gmatch (vers, "%d+") do
		table.insert(numVers, tonumber(n))
	end
	return numVers[1], numVers[2], numVers[3]
end

function MR_Utilities:RgbaToHex(r, g, b, a)
    return string.format("#%02X%02X%02X%02X", r, g, b, a)
end

function MR_Utilities:HexToRgba(hex)
    hex = hex:gsub("#","")
    if #hex == 8 then
        return tonumber("0x" .. hex:sub(1,2)),
               tonumber("0x" .. hex:sub(3,4)),
               tonumber("0x" .. hex:sub(5,6)),
               tonumber("0x" .. hex:sub(7,8))
    elseif #hex == 6 then
        return tonumber("0x" .. hex:sub(1,2)),
               tonumber("0x" .. hex:sub(3,4)),
               tonumber("0x" .. hex:sub(5,6)),
               255
    end
end

function MR_Utilities:StringToTable(str, tbl, type)
	local pattern = "(%d+) ([^,]*),"
	for _, v in str:gmatch(pattern) do
		local value = v
		if type == 'n' then
			value = tonumber(v)
		elseif type == 's' then
			if #v == 0 then
				value = ""
			end
		elseif type == 'b' then
			if v == 'true' then
				value = true
			elseif v == 'false' then
				value = false
			elseif v == 'nil' then
				value = nil
			end
		end
		table.insert(tbl, value)
	end
end

function MR_Utilities:TableToString(tbl)
	local str = ''
	for i, v in pairs(tbl) do
		local text = tostring(v)
		str = str..#text..' '..text..','
	end

	return str
end

function MR_Utilities:DistributeValuesToTables(resultTables, values, groupSizes)
	local currentIndex = 1
    for i, size in ipairs(groupSizes) do
        local group = {}
        for j = 1, size do
            if values[currentIndex] then
                table.insert(group, values[currentIndex])
                currentIndex = currentIndex + 1
            else
                break
            end
        end
        resultTables[i] = group
    end
end

function MR_Utilities:GenerateDistinctColor(colorInput)
	local hueList = {}
    if type(colorInput) == 'table' then
        for _, color in ipairs(colorInput) do
            local h, _, _ = MR_Utilities:RgbToHsb(color.r, color.g, color.b)
            table.insert(hueList, h)
        end
    else
        local h, _, _ = MR_Utilities:RgbToHsb(colorInput.r, colorInput.g, colorInput.b)
        hueList = {h}
    end

    local newHue, saturation, brightness
    local attempt = 0
    repeat
        newHue = math.random(0, 360)
        saturation = 100  -- Always use full saturation
        brightness = math.random(85, 100)
        attempt = attempt + 1
        local isTooSimilar = false
        for _, hue in ipairs(hueList) do
            local diff = math.abs(newHue - hue)
            if diff < 10 or diff > 350 then  -- Ensure hues are not too close
                isTooSimilar = true
                break
            end
        end
    until not isTooSimilar or attempt > 250

    local r, g, b = MR_Utilities:HsbToRgb(newHue, saturation, brightness)
    local newColor = LM.rgb_color:new_local()
    newColor.r = r
    newColor.g = g
    newColor.b = b
    newColor.a = 255

    return newColor
end

function MR_Utilities:RgbToHsb(r, g, b)
    r, g, b = r / 255, g / 255, b / 255
    local max, min = math.max(r, g, b), math.min(r, g, b)
    local h, s, v = max, max, max

    local d = max - min
    s = max == 0 and 0 or d / max

    if max == min then
        h = 0 -- achromatic
    else
        if max == r then
            h = (g - b) / d + (g < b and 6 or 0)
        elseif max == g then
            h = (b - r) / d + 2
        elseif max == b then
            h = (r - g) / d + 4
        end
        h = h / 6
    end
    return h * 360, s * 100, v * 100 -- Convert to degrees and percentages
end

function MR_Utilities:HsbToRgb(h, s, v)
    h = (h % 360) / 360
    s = s / 100
    v = v / 100
    local i = math.floor(h * 6)
    local f = h * 6 - i
    local p = v * (1 - s)
    local q = v * (1 - f * s)
    local t = v * (1 - (1 - f) * s)
    i = i % 6

    local r, g, b
    if i == 0 then r, g, b = v, t, p
    elseif i == 1 then r, g, b = q, v, p
    elseif i == 2 then r, g, b = p, v, t
    elseif i == 3 then r, g, b = p, q, v
    elseif i == 4 then r, g, b = t, p, v
    elseif i == 5 then r, g, b = v, p, q
    end

    return math.floor(r * 255), math.floor(g * 255), math.floor(b * 255)
end

function MR_Utilities:RandomizeBrightness(originalColor)
    local h, s, v = MR_Utilities:RgbToHsb(originalColor.r, originalColor.g, originalColor.b)

    local brightnessAdjustment = v * (0.3 * math.random() * (math.random() < 0.5 and -1 or 1))
    local newBrightness = math.max(0, math.min(100, v + brightnessAdjustment))

    local r, g, b = MR_Utilities:HsbToRgb(h, s, newBrightness)

    local newColor = LM.rgb_color:new_local()
    newColor.r = r
    newColor.g = g
    newColor.b = b
    newColor.a = originalColor.a

    return newColor
end

function MR_Utilities:FileExists(path)
    local file = io.open(path, "r")
    if file then
        file:close()
        return true
    else
        return false
    end
end

function MR_Utilities:Compare(a, b)
	return a > b 
end

function MR_Utilities:GetDirection(vec1, vec2, vec3)
	return (vec2.x - vec1.x) * (vec3.y - vec2.y) - (vec2.y - vec1.y) * (vec3.x - vec2.x)
end

function MR_Utilities:GetAngle(vec1, vec2)
	return math.atan2(vec2.y - vec1.y, vec2.x - vec1.x)
end

function MR_Utilities:ConvertPathToLuaString(path)
    local escapedPath = path:gsub("\\", "\\\\"):gsub("\"", "\\\"")
    return "\"" .. escapedPath .. "\""
end

function MR_Utilities:ConvertLuaStringToPath(luaString)
    if not luaString or luaString == "" then
        return ""
    end

    local path = luaString:gsub("\\+", "/")
    if path:sub(1,1) == "\"" and path:sub(-1) == "\"" then
        path = path:sub(2, -2)
    end

    return path
end

function MR_Utilities:DrawRectangle(g, center, sizeH, sizeV, roundRD, roundLD, roundLU, roundRU, roundInterpolation, angle, fill, line)
	local interpolationRD = MR_Utilities:CalculateInterpolation(roundRD, math.min(sizeH, sizeV), roundInterpolation)
	local interpolationLD = MR_Utilities:CalculateInterpolation(roundLD, math.min(sizeH, sizeV), roundInterpolation)
	local interpolationLU = MR_Utilities:CalculateInterpolation(roundLU, math.min(sizeH, sizeV), roundInterpolation)
	local interpolationRU = MR_Utilities:CalculateInterpolation(roundRU, math.min(sizeH, sizeV), roundInterpolation)
	
	if roundRD == 0 then
		interpolationRD = 0
	end
	if roundLD == 0 then
		interpolationLD = 0
	end
	if roundLU == 0 then
		interpolationLU = 0
	end
	if roundRU == 0 then
		interpolationRU = 0
	end
	
	local v1 = LM.Vector2:new_local()
	local v2 = LM.Vector2:new_local()
	local vT1 = LM.Vector2:new_local()
	local vT2 = LM.Vector2:new_local()
	local cornerCenter = LM.Vector2:new_local()
	
	if fill then
		g:BeginShape()
		v1:Set(center.x + (sizeH / 2), center.y + (sizeV / 2) - roundRU)
		v2:Set(v1.x, center.y - (sizeV / 2) + roundRD)
		vT1:Set(self:RotateVector2(v1, center, angle))
		vT2:Set(self:RotateVector2(v2, center, angle))
		
		g:AddLine(vT1, vT2)

		-- corner RD
		cornerCenter:Set(v2.x - roundRD, v2.y)
		v2:Set(cornerCenter.x + roundRD, cornerCenter.y)
		for i=1, interpolationRD do
			v1:Set(v2.x, v2.y)
			v2:Set(self:RotateVector2(v1, cornerCenter, math.rad(-90 / interpolationRD)))
			vT1:Set(self:RotateVector2(v1, center, angle))
			vT2:Set(self:RotateVector2(v2, center, angle))
			
			g:AddLine(vT1, vT2)
		end
		
		v1:Set(v2.x, v2.y)
		v2:Set(v1.x - sizeH + roundLD + roundRD, v1.y)
		vT1:Set(self:RotateVector2(v1, center, angle))
		vT2:Set(self:RotateVector2(v2, center, angle))
		
		g:AddLine(vT1, vT2)
		
		-- corner LD
		cornerCenter:Set(v2.x, v2.y + roundLD)
		v2:Set(cornerCenter.x, cornerCenter.y - roundLD)
		for i=1, interpolationLD do
			v1:Set(v2.x, v2.y)
			v2:Set(self:RotateVector2(v1, cornerCenter, math.rad(-90 / interpolationLD)))
			vT1:Set(self:RotateVector2(v1, center, angle))
			vT2:Set(self:RotateVector2(v2, center, angle))
			
			g:AddLine(vT1, vT2)
		end
		
		v1:Set(v2.x, v2.y)
		v2:Set(v1.x, v1.y  + sizeV - roundLD - roundLU)
		vT1:Set(self:RotateVector2(v1, center, angle))
		vT2:Set(self:RotateVector2(v2, center, angle))
		
		g:AddLine(vT1, vT2)
		
		-- corner LU
		cornerCenter:Set(v2.x + roundLU, v2.y)
		v2:Set(cornerCenter.x - roundLU, cornerCenter.y)
		for i=1, interpolationLU do
			v1:Set(v2.x, v2.y)
			v2:Set(self:RotateVector2(v1, cornerCenter, math.rad(-90 / interpolationLU)))
			vT1:Set(self:RotateVector2(v1, center, angle))
			vT2:Set(self:RotateVector2(v2, center, angle))
			
			g:AddLine(vT1, vT2)
		end
		
		v1:Set(v2.x, v2.y)
		v2:Set(v1.x + sizeH - roundRU - roundLU, v1.y)
		vT1:Set(self:RotateVector2(v1, center, angle))
		vT2:Set(self:RotateVector2(v2, center, angle))
		
		g:AddLine(vT1, vT2)
		
		-- corner RU
		cornerCenter:Set(v2.x, v2.y - roundRU)
		v2:Set(cornerCenter.x, cornerCenter.y + roundRU)
		for i=1, interpolationRU do
			v1:Set(v2.x, v2.y)
			v2:Set(self:RotateVector2(v1, cornerCenter, math.rad(-90 / interpolationRU)))
			vT1:Set(self:RotateVector2(v1, center, angle))
			vT2:Set(self:RotateVector2(v2, center, angle))
			
			g:AddLine(vT1, vT2)
		end

		g:EndShape()
	end
	
	if line then
		v1:Set(center.x + (sizeH / 2), center.y + (sizeV / 2) - roundRU)
		v2:Set(v1.x, center.y - (sizeV / 2) + roundRD)
		vT1:Set(self:RotateVector2(v1, center, angle))
		vT2:Set(self:RotateVector2(v2, center, angle))
		g:DrawLine(vT1.x, vT1.y, vT2.x, vT2.y)

		-- corner RD
		cornerCenter:Set(v2.x - roundRD, v2.y)
		v2:Set(cornerCenter.x + roundRD, cornerCenter.y)
		for i=1, interpolationRD do
			v1:Set(v2.x, v2.y)
			v2:Set(self:RotateVector2(v1, cornerCenter, math.rad(-90 / interpolationRD)))
			vT1:Set(self:RotateVector2(v1, center, angle))
			vT2:Set(self:RotateVector2(v2, center, angle))
			g:DrawLine(vT1.x, vT1.y, vT2.x, vT2.y)
		end
		
		v1:Set(v2.x, v2.y)
		v2:Set(v1.x - sizeH + roundLD + roundRD, v1.y)
		vT1:Set(self:RotateVector2(v1, center, angle))
		vT2:Set(self:RotateVector2(v2, center, angle))
		g:DrawLine(vT1.x, vT1.y, vT2.x, vT2.y)
		
		-- corner LD
		cornerCenter:Set(v2.x, v2.y + roundLD)
		v2:Set(cornerCenter.x, cornerCenter.y - roundLD)
		for i=1, interpolationLD do
			v1:Set(v2.x, v2.y)
			v2:Set(self:RotateVector2(v1, cornerCenter, math.rad(-90 / interpolationLD)))
			vT1:Set(self:RotateVector2(v1, center, angle))
			vT2:Set(self:RotateVector2(v2, center, angle))
			g:DrawLine(vT1.x, vT1.y, vT2.x, vT2.y)
		end
		
		v1:Set(v2.x, v2.y)
		v2:Set(v1.x, v1.y  + sizeV - roundLD - roundLU)
		vT1:Set(self:RotateVector2(v1, center, angle))
		vT2:Set(self:RotateVector2(v2, center, angle))
		g:DrawLine(vT1.x, vT1.y, vT2.x, vT2.y)
		
		-- corner LU
		cornerCenter:Set(v2.x + roundLU, v2.y)
		v2:Set(cornerCenter.x - roundLU, cornerCenter.y)
		for i=1, interpolationLU do
			v1:Set(v2.x, v2.y)
			v2:Set(self:RotateVector2(v1, cornerCenter, math.rad(-90 / interpolationLU)))
			vT1:Set(self:RotateVector2(v1, center, angle))
			vT2:Set(self:RotateVector2(v2, center, angle))
			g:DrawLine(vT1.x, vT1.y, vT2.x, vT2.y)
		end
		
		v1:Set(v2.x, v2.y)
		v2:Set(v1.x + sizeH - roundRU - roundLU, v1.y)
		vT1:Set(self:RotateVector2(v1, center, angle))
		vT2:Set(self:RotateVector2(v2, center, angle))
		g:DrawLine(vT1.x, vT1.y, vT2.x, vT2.y)
		
		-- corner RU
		cornerCenter:Set(v2.x, v2.y - roundRU)
		v2:Set(cornerCenter.x, cornerCenter.y + roundRU)
		for i=1, interpolationRU do
			v1:Set(v2.x, v2.y)
			v2:Set(self:RotateVector2(v1, cornerCenter, math.rad(-90 / interpolationRU)))
			vT1:Set(self:RotateVector2(v1, center, angle))
			vT2:Set(self:RotateVector2(v2, center, angle))
			g:DrawLine(vT1.x, vT1.y, vT2.x, vT2.y)
		end
	end
end

function MR_Utilities:CalculateInterpolation(round, size, factor)
    local baseInterpolation = 3
    local interpolationFactor = 45 * factor
    local maxIterations = 30
    
    local interpolation = baseInterpolation + math.floor(math.log(1 + round / size) * interpolationFactor)
    return math.min(maxIterations, math.max(baseInterpolation, interpolation))
end

function MR_Utilities:SortTableAlphabetically(tbl, lettersFirst)
	if lettersFirst then
		 table.sort(tbl, function(a, b)

        local cleanA = a:match("%a") and a:match("%a.*") or a
        local cleanB = b:match("%a") and b:match("%a.*") or b

        if cleanA == "" then
            return false
        elseif cleanB == "" then
            return true
        elseif cleanA:match("^%a") and not cleanB:match("^%a") then
            return true
        elseif not cleanA:match("^%a") and cleanB:match("^%a") then
            return false
        else
            return cleanA:lower() < cleanB:lower()
        end
    end)
	else
		table.sort(tbl, function(a, b)
			if a == "" then
				return false
			elseif b == "" then
				return true
			else
				return a:lower() < b:lower()
			end
		end)
	end
end

function MR_Utilities:ReplaceTextInString(originalString, searchText, replaceText)
    local resultString = originalString:gsub(searchText, replaceText)
    return resultString
end

function MR_Utilities:GetOffsetChannel(moho, layer, curve, curvePoint, prePoint)
	-- This part of the code is taken from A. Evseeva's AE_Utilities script
	local mesh = moho:LayerAsVector(layer)
	if not mesh then return nil end
	mesh = mesh:Mesh()
	local ptID = mesh:PointID(curve:Point(curvePoint))
	local crvID = mesh:CurveID(curve)
	local channelOffset = 3
	if not prePoint then channelOffset = 4 end
	local subNo = 0
	for p = 0, (ptID - 1) do
		subNo = subNo + mesh:Point(p):CountCurves()*5
	end
	for c = 0, mesh:Point(ptID):CountCurves() - 1 do
		local nextCurve = mesh:Point(ptID):Curve(c,-1)
		if nextCurve == curve then break
		else subNo = subNo + 5
		end
	end
	subNo = subNo + channelOffset
	-- End of the code taken from A. Evseeva's script
	
	local chInfo = MOHO.MohoLayerChannel:new_local()
	for i=0, layer:CountChannels()-1 do
		layer:GetChannelInfo(i, chInfo)
		if chInfo.channelID == CHANNEL_CURVE then
			return moho:ChannelAsAnimVal(layer:Channel(i, subNo, moho.document))
		end
	end
end

function MR_Utilities:GetWeightChannel(moho, layer, curve, curvePoint, prePoint)
	-- This part of the code is taken from A. Evseeva's AE_Utilities script
	local mesh = moho:LayerAsVector(layer)
	if not mesh then return nil end
	mesh = mesh:Mesh()
	local ptID = mesh:PointID(curve:Point(curvePoint))
	local crvID = mesh:CurveID(curve)
	local channelWeight = 1
	if not prePoint then channelWeight = 2 end
	local subNo = 0
	for p = 0, (ptID - 1) do
		subNo = subNo + mesh:Point(p):CountCurves()*5
	end
	for c = 0, mesh:Point(ptID):CountCurves() - 1 do
		local nextCurve = mesh:Point(ptID):Curve(c,-1)
		if nextCurve == curve then break
		else subNo = subNo + 5
		end
	end
	subNo = subNo + channelWeight
	-- End of the code taken from A. Evseeva's script

	local chInfo = MOHO.MohoLayerChannel:new_local()
	for i=0, layer:CountChannels()-1 do
		layer:GetChannelInfo(i, chInfo)
		if chInfo.channelID == CHANNEL_CURVE then
			return moho:ChannelAsAnimVal(layer:Channel(i, subNo, moho.document))
		end
	end
end

function MR_Utilities:ScaleVector2(vector2, center, scaling)
	local centerVec = LM.Vector2:new_local()
	centerVec:Set(center)
	local dif = LM.Vector2:new_local()
	local newVector2 = LM.Vector2:new_local()
	dif:Set(vector2 - centerVec)
	
	if type(scaling) == "number" then
		newVector2:Set((dif * scaling) + centerVec)
	else
		newVector2:Set(dif.x * scaling.x + centerVec.x, dif.y * scaling.y + centerVec.y)
	end
	
	return newVector2
end

function MR_Utilities:IsLayerVisible(moho, layer, skel, ignoreMasks)
	local layerVisibility = true

	if skel then
		local parentBoneID = layer:LayerParentBone()
		if parentBoneID == -1 then
			for i=0, skel:CountGroups()-1 do
				local group = skel:Group(i)
				for b=0, group:CountBones()-1 do
					local bone = group:Bone(b)
					if not bone:IsGroupVisible() then
						if layer:IsIncludedInFlexiBoneSubset(skel:BoneID(bone)) then
							return false
						end
					end
				end
			end
		end

		if parentBoneID >= 0 then
			local skelL = layer:ControllingSkeleton()
			if skelL then
				local bone = skelL:Bone(parentBoneID)
				if not bone:IsGroupVisible() then
					return false
				end
			end
		end
	end

	if not layer.fVisibility.value or not layer:IsVisible() then
		layerVisibility = false
	elseif layer:IsRenderOnly() then
		layerVisibility = false
	elseif layer:MaskingMode() == 5 and not ignoreMasks then
		layerVisibility = false
	else
		local targetLayer = layer:Parent()
		local lastLayer = layer
		if targetLayer then
			repeat
				if not targetLayer.fVisibility.value or not targetLayer:IsVisible() then
					layerVisibility = false
					break
				end

				local skel = targetLayer:ControllingSkeleton()

				if moho:LayerAsBone(targetLayer) then
					local parentLayer = targetLayer:Parent()
					if parentLayer then
						skel = parentLayer:ControllingSkeleton()
					end
				end

				if skel then
					local parentBoneID = targetLayer:LayerParentBone()
					if parentBoneID >= 0 then
						local bone = skel:Bone(parentBoneID)
						if not bone:IsGroupVisible() then
							layerVisibility = false
							break
						end
					end
				end

				if targetLayer:LayerType() == MOHO.LT_SWITCH then
					local switchLayer = moho:LayerAsSwitch(targetLayer)
					local switchChannel = switchLayer:SwitchValues()
					local switchValue = switchChannel.value:Buffer()

					local isSwitchValueCorrect = false
					for i=0, targetLayer:CountLayers() - 1 do
						if targetLayer:Layer(i):Name() == switchValue then
							isSwitchValueCorrect = true
							break
						end
					end

					if switchValue ~= lastLayer:Name() then
						if not isSwitchValueCorrect then
							local targetSwitchLayer = switchLayer:Layer(switchLayer:CountLayers() - 1):Name()
							if targetSwitchLayer ~= lastLayer:Name() then
								layerVisibility = false
								break
							end
						else
							layerVisibility = false
							break
						end
					end
				end
				lastLayer = targetLayer
				targetLayer = targetLayer:Parent()
			until targetLayer == nil
		end
	end

	if layerVisibility then
		local skel = layer:ControllingSkeleton()
		if not skel then
			return layerVisibility
		end

		local IsInvisibleBoneInSubset = false
		local IsVisibleBoneInSubset = false
		local IsParentBone = false
		local IsParentBoneVisible = false
		for i=0, skel:CountBones()-1 do
			local bone = skel:Bone(i)
			if layer:IsIncludedInFlexiBoneSubset(i) then
				if bone:IsGroupVisible() then
					IsVisibleBoneInSubset = true
				else
					IsInvisibleBoneInSubset = true
				end
			end
		end
		if IsInvisibleBoneInSubset and not IsVisibleBoneInSubset then
			layerVisibility = false
		end
	end

	return layerVisibility
end

function MR_Utilities:DrawInProgressIcon(moho, view, g, centerVec, radius)
	local m = LM.Matrix:new_local()
	local viewAngle = g:ViewRotation()
	m:Rotate(2, -viewAngle)

	g:Push()
	g:ApplyMatrix(m)
	local currentScale = g:CurrentScale(false)
	local height = g:Height() / 1080
	local center = LM.Vector2:new_local()
	center:Set(centerVec)
	m:Invert()
	m:Transform(center)

	local radiusCorrected = radius / currentScale / height

	local shadowOffset = LM.Vector2:new_local()
	shadowOffset:Set(0.002 / currentScale / height, -0.005 / currentScale / height)
	local v = LM.Vector2:new_local()
	g:SetSmoothing(true)

	g:SetPenWidth(10)
	g:SetColor(0, 0, 0, 50)

	self:DrawArrow(g, center + shadowOffset, radiusCorrected, 105 - viewAngle, -90, 45, 5)

	g:SetPenWidth(10)
	g:SetColor(255, 255, 255, 255)

	self:DrawArrow(g, center, radiusCorrected, 105 - viewAngle, -90, 45, 5)

	g:SetPenWidth(10)
	g:SetColor(0, 0, 0, 50)

	self:DrawArrow(g, center + shadowOffset, radiusCorrected, 285 - viewAngle, -90, 45, 5)

	g:SetPenWidth(10)
	g:SetColor(255, 255, 255, 255)

	self:DrawArrow(g, center, radiusCorrected, 285 - viewAngle, -90, 45, 5)

	g:Pop()
end

function MR_Utilities:DrawArrow(g, center, outerRadius, angleOffset, startAngleDeg, endAngleDeg, stepDeg)
	g:BeginShape()
	local outerArcPoints = {}
	local angleStep = stepDeg or 5
	local function condition(angle)
		if startAngleDeg + angleOffset < endAngleDeg + angleOffset then
			return angle <= endAngleDeg + angleOffset
		else
			return angle >= endAngleDeg + angleOffset
		end
	end
	local angle = startAngleDeg + angleOffset
	while condition(angle) do
		local rad = math.rad(angle)
		local x = center.x + outerRadius * math.cos(rad)
		local y = center.y + outerRadius * math.sin(rad)
		local vec = LM.Vector2:new_local()
		vec:Set(x, y)
		table.insert(outerArcPoints, vec)
		if startAngleDeg + angleOffset < endAngleDeg + angleOffset then
			angle = angle + angleStep
		else
			angle = angle - angleStep
		end
	end
	local rad = math.rad(endAngleDeg + angleOffset)
	local x = center.x + outerRadius * math.cos(rad)
	local y = center.y + outerRadius * math.sin(rad)
	local vec = LM.Vector2:new_local()
	vec:Set(x, y)
	table.insert(outerArcPoints, vec)
	for i = 1, (#outerArcPoints - 1) do
		g:AddLine(outerArcPoints[i], outerArcPoints[i+1])
	end
	local arcEnd = outerArcPoints[#outerArcPoints]
	local arcEndPrev = outerArcPoints[#outerArcPoints - 1] or arcEnd
	local dx = arcEnd.x - arcEndPrev.x
	local dy = arcEnd.y - arcEndPrev.y
	local arrowAngleDeg = math.deg(math.atan2(dy, dx))
	local tipLength = outerRadius * 0.5
	local halfWidth = tipLength * 0.275
	local function offsetPoint(base, angleDeg, dist)
		local p = LM.Vector2:new_local()
		local newX = base.x + dist * math.cos(math.rad(angleDeg))
		local newY = base.y + dist * math.sin(math.rad(angleDeg))
		p:Set(newX, newY)
		return p
	end
	local tipTop = offsetPoint(arcEnd, arrowAngleDeg + 45 + angleOffset, halfWidth)
	local tipFront = offsetPoint(tipTop, arrowAngleDeg + 170 + angleOffset, tipLength)
	local tipBottom = offsetPoint(tipFront, arrowAngleDeg + 280 + angleOffset, tipLength)
	g:AddLine(arcEnd, tipTop)
	g:AddLine(tipTop, tipFront)
	g:AddLine(tipFront, tipBottom)

	local innerRadius = outerRadius * 0.7
	local innerArcPoints = {}
	angle = endAngleDeg + angleOffset
	while true do
		local rad = math.rad(angle)
		local x = center.x + innerRadius * math.cos(rad)
		local y = center.y + innerRadius * math.sin(rad)
		local vec = LM.Vector2:new_local()
		vec:Set(x, y)
		table.insert(innerArcPoints, vec)
		if startAngleDeg + angleOffset < endAngleDeg + angleOffset then
			if angle <= startAngleDeg + angleOffset then break end
			angle = angle - angleStep
		else
			if angle >= startAngleDeg + angleOffset then break end
			angle = angle + angleStep
		end
	end

	g:AddLine(tipBottom, innerArcPoints[1])
	local rad2 = math.rad(startAngleDeg + angleOffset)
	local x2 = center.x + innerRadius * math.cos(rad2)
	local y2 = center.y + innerRadius * math.sin(rad2)
	local vec2 = LM.Vector2:new_local()
	vec2:Set(x2, y2)
	table.insert(innerArcPoints, vec2)
	for i = 1, (#innerArcPoints - 1) do
		g:AddLine(innerArcPoints[i], innerArcPoints[i+1])
	end
	g:AddLine(innerArcPoints[#innerArcPoints], outerArcPoints[1])
	g:EndShape()
end

function MR_Utilities:CheckFlipDirection(moho, layer, point)
	if not point or not layer then
		return
	end
	local flipDirection = false
	local flipDirectionV = false
	local curve, where = point:Curve(0, -1)
	local parent = point.fParent

	local originalOffset = MR_Utilities:GetOffsetChannel(moho, layer, curve, where, true).value
	point.fParent = -1
	layer:UpdateCurFrame()
	local newOffset = MR_Utilities:GetOffsetChannel(moho, layer, curve, where, true).value
	point.fParent = parent
	if originalOffset ~= newOffset or (tostring(originalOffset) == '-0.0' and tostring(newOffset) == '0.0') or (tostring(originalOffset) == '0.0' and tostring(newOffset) == '-0.0') then
		flipDirection = true
	end
	layer:UpdateCurFrame()
	return flipDirection
end

function MR_Utilities:UpdatePanel(moho)
	local drawingToolsNonZero = MOHO.MohoGlobals.DisableDrawingToolsNonZero
	if not drawingToolsNonZero then
		MOHO.MohoGlobals.DisableDrawingToolsNonZero = true
	end
	local frame = moho.frame
	local curLayer = moho.layer
	if moho.layer:LayerType() ~= MOHO.LT_BONE and moho.layer:LayerType() ~= MOHO.LT_VECTOR then
		local count = 0
		repeat
			local layer = moho.document:LayerByAbsoluteID(count)
			if layer then
				count = count + 1
				if layer:LayerType() == MOHO.LT_BONE or layer:LayerType() == MOHO.LT_BONE then
					moho:SetSelLayer(layer)
					break
				end
			end
		until not layer
	end

	if frame == 0 then
		moho:SetCurFrame(1)
		moho:SetSelLayer(curLayer)
		moho:SetCurFrame(0)
	elseif frame ~= 0 then
		moho:SetCurFrame(0)
		moho:SetSelLayer(curLayer)
		moho:SetCurFrame(frame)
	end

	if not drawingToolsNonZero then
		MOHO.MohoGlobals.DisableDrawingToolsNonZero = drawingToolsNonZero
	end
end

function MR_Utilities:IsLayerActive(moho, layer, skel)
	local layerVisibility = true

	if skel then
		local parentBoneID = layer:LayerParentBone()
		if parentBoneID == -1 then
			for i=0, skel:CountGroups()-1 do
				local group = skel:Group(i)
				for b=0, group:CountBones()-1 do
					local bone = group:Bone(b)
					if not bone:IsGroupVisible() then
						if layer:IsIncludedInFlexiBoneSubset(skel:BoneID(bone)) then
							return false
						end
					end
				end
			end
		end

		if parentBoneID >= 0 then
			local skelL = layer:ControllingSkeleton()
			if skelL then
				local bone = skelL:Bone(parentBoneID)
				if not bone:IsGroupVisible() then
					return false
				end
			end
		end
	end


	local targetLayer = layer:Parent()
	local lastLayer = layer
	if targetLayer then
		repeat
			local skel = targetLayer:ControllingSkeleton()

			if moho:LayerAsBone(targetLayer) then
				local parentLayer = targetLayer:Parent()
				if parentLayer then
					skel = parentLayer:ControllingSkeleton()
				end
			end

			if skel then
				local parentBoneID = targetLayer:LayerParentBone()
				if parentBoneID >= 0 then
					local bone = skel:Bone(parentBoneID)
					if not bone:IsGroupVisible() then
						layerVisibility = false
						break
					end
				end
			end

			if targetLayer:LayerType() == MOHO.LT_SWITCH then
				local switchLayer = moho:LayerAsSwitch(targetLayer)
				local switchChannel = switchLayer:SwitchValues()
				local switchValue = switchChannel.value:Buffer()

				local isSwitchValueCorrect = false
				for i=0, targetLayer:CountLayers() - 1 do
					if targetLayer:Layer(i):Name() == switchValue then
						isSwitchValueCorrect = true
						break
					end
				end

				if switchValue ~= lastLayer:Name() then
					if not isSwitchValueCorrect then
						local targetSwitchLayer = switchLayer:Layer(switchLayer:CountLayers() - 1):Name()
						if targetSwitchLayer ~= lastLayer:Name() then
							layerVisibility = false
							break
						end
					else
						layerVisibility = false
						break
					end
				end
			end
			lastLayer = targetLayer
			targetLayer = targetLayer:Parent()
		until targetLayer == nil
	end

	if layerVisibility then
		local skel = layer:ControllingSkeleton()
		if not skel then
			return layerVisibility
		end

		local IsInvisibleBoneInSubset = false
		local IsVisibleBoneInSubset = false
		local IsParentBone = false
		local IsParentBoneVisible = false
		for i=0, skel:CountBones()-1 do
			local bone = skel:Bone(i)
			if layer:IsIncludedInFlexiBoneSubset(i) then
				if bone:IsGroupVisible() then
					IsVisibleBoneInSubset = true
				else
					IsInvisibleBoneInSubset = true
				end
			end
		end
		if IsInvisibleBoneInSubset and not IsVisibleBoneInSubset then
			layerVisibility = false
		end
	end

	return layerVisibility
end